const bcrypt = require('bcrypt');
const crypto = require("crypto");
const axios = require("axios");
const jwt = require('jsonwebtoken');
const config = require('../../configs/jwt.configs.js');
const sendEmail = require("../utils/sendEmail");

const { HTTP_SUCCESS_RESPONSE } = require("../utils/success");
const { HTTP400Error, HTTP404Error, catchAsync } = require("../utils/errors");
const XLSX = require("xlsx");

const handlebars = require("handlebars");
const fs = require("fs");
const path = require("path");
const resetPasswordTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/resetPassword.handlebars"), "utf-8"); 
const verifyEmailTemplate = fs.readFileSync(path.join(__dirname, "../utils/templates/verifyEmail.handlebars"), "utf-8"); 
// Import model
const Driver = require('../models/drivers.js');

const saveAddressCoordinates = async (address) => {
    return await axios(`https://maps.googleapis.com/maps/api/geocode/json?address=${encodeURIComponent(address)}&key=${process.env.GOOGLE_MAP_API_KEY}`)
    .then((response) => {
        return {status:200, data: response.data.results[0] }
    })
    .catch((error) => {
        console.log(error);
        return {status:400, data: error.response }
    })
}

module.exports = {
    create: catchAsync(async (req, res) => {
        const driverData = { firstName, lastName, email, phone, password, physicalAddress, mailingAddress, isNotify } = req.body;
        let hashedPassword = bcrypt.hashSync(password, 8)
        let existDriver = await Driver.find({ email: email});
        if(existDriver.length < 1) {
            let driver = new Driver(driverData);
            driver.password = hashedPassword;
            return driver.save()
            .then(async function(driver) {
                let address = `${driver.physicalAddress.street}, ${driver.physicalAddress.city}, ${driver.physicalAddress.state}, ${driver.physicalAddress.zip}`;
                let coordinatesData = await saveAddressCoordinates(address);
                if(coordinatesData.status == 200) {
                    driver.addressCoordinates ={ latitude: coordinatesData.data.geometry?coordinatesData.data.geometry.location.lat:'', longitude: coordinatesData.data.geometry?coordinatesData.data.geometry.location.lng:'', place_id: coordinatesData.data.place_id?coordinatesData.data.place_id:'' }
                }
                driver.token = crypto.randomBytes(32).toString("hex");

                const template = handlebars.compile(verifyEmailTemplate);
                let message = (template({
                    verifyEmailLink: `${process.env.API_BASE_URL}/api/v1/driver/users/verify/${driver._id}/${driver.token}`,
                }));

                sendEmail(driver.email, "Verify Your Email", message);

                await driver.setCreatedBy(driver._id); 
                await driver.setUpdatedBy(driver._id);
                await driver.save();
                return HTTP_SUCCESS_RESPONSE(res, { data: driver }, 'Driver User Successfully Created.' ); 
            }).catch(err => {
                throw new HTTP400Error(err.message);
            });    
            
        } else {
            throw new HTTP400Error('Eamil Already Exists');
        }
    }),

    login: catchAsync(async(req, res) => {
        const { email, password } = req.body;
        let driver = await Driver.findOne({ email: email });
        if(!driver ) throw new HTTP404Error('Email Does Not Exist.');
        driver = driver.toJSON(); 
        let passwordIsValid = await bcrypt.compareSync(password, driver.password);
        if (!passwordIsValid) throw new HTTP400Error('Password does not matched.');

        if(!driver.isVerified) throw new HTTP400Error('Your email is not verified. Please verify.');

        let token = jwt.sign({ _id: driver._id, email: driver.email }, config.secret, { expiresIn: '60m' });
        let refreshToken = jwt.sign({ _id: driver._id, email: driver.email }, config.secret, { expiresIn: '120m' });

        res.setHeader("Authorization", token);
        return HTTP_SUCCESS_RESPONSE(res, { token: token, refreshToken: refreshToken  }, 'Auth Successful' );
    }),

    googleLogin: catchAsync(async(req, res) => {
        const { firstName, lastName, email, profileImg } = req.body;
        let driver = await Driver.findOne({ email: email });
        if(!driver ) {
            let newDriver = new Driver();
            newDriver.firstName = firstName;
            newDriver.lastName = lastName;
            newDriver.email = email;
            newDriver.profileImg = profileImg;
            newDriver.isVerified = true;
            newDriver.isNotify = true;
            newDriver.accountType = 'google';
            newDriver.save()
            .then ((driverData) => {
                let token = jwt.sign({ _id: driverData._id, email: driverData.email }, config.secret, { expiresIn: '60m' });
                let refreshToken = jwt.sign({ _id: driverData._id, email: driverData.email }, config.secret, { expiresIn: '120m' });
                res.setHeader("Authorization", token);
                return HTTP_SUCCESS_RESPONSE(res, { data: driverData, token: token, refreshToken: refreshToken  }, 'Auth Successful' );
            })
            .catch((error) => {
                console.log(error);
                throw new HTTP400Error(error);
            })
        } else {
            driver.firstName = firstName;
            driver.lastName = lastName;
            driver.email = email;
            driver.profileImg = profileImg;
            driver.save();
            let token = jwt.sign({ _id: driver._id, email: driver.email }, config.secret, { expiresIn: '60m' });
            let refreshToken = jwt.sign({ _id: driver._id, email: driver.email }, config.secret, { expiresIn: '120m' });
            res.setHeader("Authorization", token);
            return HTTP_SUCCESS_RESPONSE(res, { data: driver, token: token, refreshToken: refreshToken  }, 'Auth Successful' );
        }
    }),

    facebookLogin: catchAsync(async(req, res) => {
        const { firstName, lastName, email, profileImg } = req.body;
        let driver = await Driver.findOne({ email: email });
        if(!driver ) {
            let newDriver = new Driver();
            newDriver.firstName = firstName;
            newDriver.lastName = lastName;
            newDriver.email = email;
            newDriver.profileImg = profileImg;
            newDriver.isVerified = true;
            newDriver.isNotify = true;
            newDriver.accountType = 'facebook';
            newDriver.save()
            .then ((driverData) => {
                let token = jwt.sign({ _id: driverData._id, email: driverData.email }, config.secret, { expiresIn: '60m' });
                let refreshToken = jwt.sign({ _id: driverData._id, email: driverData.email }, config.secret, { expiresIn: '120m' });
                res.setHeader("Authorization", token);
                return HTTP_SUCCESS_RESPONSE(res, { data: driverData, token: token, refreshToken: refreshToken  }, 'Auth Successful' );
            })
            .catch((error) => {
                console.log(error);
                throw new HTTP400Error(error);
            })
        } else {
            driver.firstName = firstName;
            driver.lastName = lastName;
            driver.email = email;
            driver.profileImg = profileImg;
            driver.save();
            let token = jwt.sign({ _id: driver._id, email: driver.email }, config.secret, { expiresIn: '60m' });
            let refreshToken = jwt.sign({ _id: driver._id, email: driver.email }, config.secret, { expiresIn: '120m' });
            res.setHeader("Authorization", token);
            return HTTP_SUCCESS_RESPONSE(res, { data: driver, token: token, refreshToken: refreshToken  }, 'Auth Successful' );
        }
    }),

    update: catchAsync(async(req, res) => {
        const driverData = { firstName, lastName, phone, physicalAddress, mailingAddress } = req.body;
        let id = req.params.id;
        let driver = await Driver.findById(id);
        if(!driver) throw new HTTP404Error('Driver not found');
        driver = driver.toJSON();
        driver = {...driver, ...driverData };  
        try {
            let updatedDriver = await Driver(driver);
            await updatedDriver.setUpdatedBy(req.driver? req.driver._id: req.admin._id); 
            await updatedDriver.setUpdatedAt();
            updatedDriver = await Driver(updatedDriver); 
            return HTTP_SUCCESS_RESPONSE(res, { data: updatedDriver }, 'Driver successfully updated' );
        } catch(err) {
            throw new HTTP400Error(err.message);
        }
    }),

    list: catchAsync(async(req, res) => {
        const { page = 1, limit = 5 } = req.query;

        let driverList = await Driver.find({})
        .select({ firstName: 1, lastName: 1, phone: 1, email: 1, physicalAddress: 1, mailingAddress: 1, createdAt: 1 })
        .sort({createdAt: -1})
        .limit(limit * 1)
        .skip((page - 1) * limit)
        .exec();

        const count = await Driver.countDocuments();
        let totalPages = Math.ceil(count / limit);
        return HTTP_SUCCESS_RESPONSE(res, { data: driverList, totalPages, page }, 'Driver list successfully retrieved.' );
    }),

    countTotalDrivers: catchAsync(async(req, res) => {
        let countDrivers = await Driver.countDocuments();
        return HTTP_SUCCESS_RESPONSE(res, { data: countDrivers }, 'Successfully count drivers.' );
    }),

    
    refresh: (req, res) => {
        let token = jwt.sign({ 
            _id: req.driver._id, 
            email: req.driver.email
            },
            config.secret, { expiresIn: '60m' }
        );
        let refreshToken = jwt.sign({ _id: req.driver._id, email: req.driver.email }, config.secret, { expiresIn: '120m' });
        res.setHeader("Authorization", token);
        return HTTP_SUCCESS_RESPONSE(res, { auth: true, token: token, refreshToken: refreshToken }, 'Refresh Request Successfull');
    },

    me: catchAsync(async(req, res) => {
        let id = req.driver._id;
        let driver = await Driver.findById(id).select({ firstName: 1, lastName: 1, phone: 1, email: 1, profileImg: 1, physicalAddress: 1, mailingAddress: 1, isNotify: 1, isVerified: 1, accountType: 1 });
        if(!driver) throw new HTTP404Error('No driver found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: driver }, 'Driver successfully retrieved');
    }),


    userDetails: catchAsync(async(req, res) => {
        let id = req.params.id;
        let driver = await Driver.findById(id).select({ firstName: 1, lastName: 1, phone: 1, email: 1, physicalAddress: 1, mailingAddress: 1 });
        if(!driver) throw new HTTP404Error('No driver found.');
        return HTTP_SUCCESS_RESPONSE(res, { data: driver }, 'Driver successfully retrieved');
    }),

    destroy: catchAsync(async(req, res) => {
        let id = req.params.id;
        let driver = await Driver.findOne({ _id: id });
        if(!driver) throw new HTTP404Error('No driver found.');
        await driver.delete();
        return HTTP_SUCCESS_RESPONSE(res, {}, 'Driver successfully deleted');
    }),

    verifyDriver: catchAsync(async(req, res) => {
        let alreadyVerifiedUrl = `${process.env.DRIVER_PANEL_BASE_URL}/email-verification-exist`;
        let successUrl = `${process.env.DRIVER_PANEL_BASE_URL}/email-verification-success`;
        let errorUrl = `${process.env.DRIVER_PANEL_BASE_URL}/email-verification-failed`;

        try {
            const driver = await Driver.findOne({ _id: req.params.id });
            if (!driver) {
                return res.redirect(errorUrl);
            }

            if(driver.isVerified) {
                return res.redirect(alreadyVerifiedUrl);
            }

            const driverByToken = await Driver.findOne({
                token: req.params.token,
            });

            if (!driverByToken) {
                return res.redirect(errorUrl);
            }

            driver.isVerified = true;
            driver.token = null;
            driver.save();
            return res.redirect(successUrl);
        } catch (error) {
            return res.redirect(errorUrl);
        }
    }),

    sendEmailVerificationLink: async (req, res) => {
        try {
            const driver = await Driver.findOne({ email: req.body.email });
            if (!driver) {
                return res.status(400).send({ message: "Driver not exist." });
            }

            driver.token = crypto.randomBytes(32).toString("hex");
            driver.save();

            const template = handlebars.compile(verifyEmailTemplate);
            let message = (template({
                verifyEmailLink: `${process.env.API_BASE_URL}/api/v1/driver/users/verify/${driver._id}/${driver.token}`,
            }));

            let sendEmailData = await sendEmail(driver.email, "Verify Your Email", message);
            console.log(sendEmailData, 'Send email data');
            if(sendEmailData.status == 200) {
                return res
                .status(200)
                .json({ message: "Successfully send verification email", success: true });
            } else {
                return res
                .status(400)
                .json({ message: sendEmailData.message, success: false });
            }
            
        } catch (error) {
            console.log(error, "dfds");
            return res.status(400).send({ message: error });
        }
    },

    forgotPassword: async (req, res) => {
        try {
            const driver = await Driver.findOne({ email: req.body.email });
            if (!driver)
                return res
                .status(400)
                .send({ message: "Provider with given email doesn't exist" });

            driver.token = crypto.randomBytes(32).toString("hex");
            await driver.save();

            const template = handlebars.compile(resetPasswordTemplate);
            let message = (template({
                resetPasswordLink: `${process.env.DRIVER_PANEL_BASE_URL}/password-reset?id=${driver._id}&&token=${driver.token}`,
            }));
        
            sendEmail(driver.email, "Password reset", message);
            return res
            .status(200)
            .json({ message: "Password reset link sent to your email.", success: true });
        } catch (error) {
            return res.status(200).send({ message: error, success: false });
        }
    },

    passwordReset: async (req, res) => {
        try {
          const driver = await Driver.findById(req.params.id);
          if (!driver)
            return res.status(400).send({ message: "Driver not exist by this Id." });
    
            const driverByToken = await Driver.findOne({
                token: req.params.token,
            });
            if (!driverByToken)
                return res.status(400).send({ message: "Invalid link or expired" });

            if(req.body.password !== req.body.confirmPassword) {
                return res.status(400).send({ message: "Password and confirm password does not matched." });
            }

            let hashedPassword = bcrypt.hashSync(req.body.password, 8);
            driver.password = hashedPassword;
            driver.token = null;
            await driver.save();
    
            return res
                .status(200)
                .json({ message: "Password Successfully Changed.", success: true });
        } catch (error) {
            return res.status(400).send({ message: error });
        }
    },

    exportExcel: async (req, res) => {
        try {
            let drivers = await Driver.find({}).select({ firstName: 1, lastName: 1,  phone: 1, email: 1, physicalAddress: 1 }).sort({createdAt: -1});
            drivers = drivers.map((item) => {
                let data = {
                    firstName: item.firstName,
                    lastName: item.lastName,
                    phone: item.phone,
                    email: item.email,
                    physicalAddress: item.physicalAddress.street + ', ' + item.physicalAddress.city + ', ' +  item.physicalAddress.state + ', ' + item.physicalAddress.zip,
                }

                return data;
            })

            console.log(drivers, 'oo');

            const worksheet = XLSX.utils.json_to_sheet(drivers);
            const workbook = XLSX.utils.book_new();
            XLSX.utils.book_append_sheet(workbook, worksheet, "Drivers");
            XLSX.utils.sheet_add_aoa(worksheet, [["First Name", "Last Name", "Phone", "Email", "Physical Address"]], { origin: "A1" });
            XLSX.writeFile(workbook, "public/export/drivers.xlsx");
            return res.status(200).json({ path: `${process.env.API_BASE_URL}/export/drivers.xlsx`, message: "Successfully Export Excel" });
        } catch(error) {
            console.log(error);
            return res.status(400).json({ message: error });
        }
    }
}
