const { validationResult } = require("express-validator");

const HttpStatusCode = {
  OK: 200,
  BAD_REQUEST: 400,
  NOT_FOUND: 404,
  INTERNAL_SERVER: 500,
};
class BaseError extends Error {
  constructor(name, httpCode, message) {
    super(message);
    this.status = "error";
    this.name = name;
    this.httpCode = httpCode;
    Error.captureStackTrace(this);
  }
}
class APIError extends BaseError {
  constructor( message = "Internal server error" ) {
    super('INTERNAL SERVER ERROR', HttpStatusCode.INTERNAL_SERVER, message);
  }
}
class HTTP400Error extends BaseError {
  constructor(message = "Bad request") {
    super("BAD REQUEST", HttpStatusCode.BAD_REQUEST, message);
  }
}

class HTTP404Error extends BaseError {
    constructor(message = "Not found") {
        super("NOT FOUND", HttpStatusCode.NOT_FOUND, message);
    }
}

const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    res.status(400).json({ ...errors, message: errors.errors[0].msg });
    return;
  }
  next();
};

const catchAsync =
  (callback) =>
    (...args) =>
      callback(...args).catch((err) => {
        if (!err.httpCode) {
          console.log({ err: err.stack });
          err = new APIError();
        }
        args[1].status(err.httpCode).json({ message: err.message });
      });
module.exports = {
  HttpStatusCode,
  BaseError,
  APIError,
  HTTP400Error,
  HTTP404Error,
  handleValidationErrors,
  catchAsync,
};
