const express = require('express');
const router = express.Router();
const { param } = require("express-validator");
const { createAdminUserValidation, updateAdminUserValidation, loginAdminUserValidation } = require("../../utils/ApiValidation/admin_users");
const { createProviderUserValidation } = require("../../utils/ApiValidation/provider_users");
const { handleValidationErrors } = require("../../utils/errors");
const adminUsersAction = require('../actions/admin_users.action');
const providerUsersAction = require('../actions/provider_users.action');
const driverUsersAction = require('../actions/driver_users.action');
const paymentAction = require('../actions/payment.action');
const adminUserMiddleware = require('../middlewares/admin_users.middleware.js');

router
	.post('/',
		createAdminUserValidation,
		handleValidationErrors,
		adminUsersAction.create
	)
	.get(
		'/', 
		adminUserMiddleware.checkToken, 
		adminUsersAction.list
	)
	.post(
		'/login',
		loginAdminUserValidation,
		handleValidationErrors,
		adminUsersAction.login
	)
	.get(
		'/refresh', 
		adminUserMiddleware.checkToken, 
		adminUsersAction.refresh
	)

	.get(
		'/me', 
		adminUserMiddleware.checkToken, 
		adminUsersAction.me
	)

	.get(
		'/:id', 
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,  
		adminUsersAction.userDetails
	)
	.put(
		'/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		updateAdminUserValidation,
		handleValidationErrors, 
		adminUsersAction.update
	)
	.delete(
		'/:id', 
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors, 
		adminUsersAction.destroy
	)

	.get(
		'/provider/approve/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		providerUsersAction.approveProviderByAdmin
	)

	.get(
		'/provider/verify-email/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		providerUsersAction.verifyProviderEmailByAdmin
	)

	.post(
		'/provider/create',
		adminUserMiddleware.checkToken,
		createProviderUserValidation,
		handleValidationErrors,
		providerUsersAction.createProviderByAdmin
	)

	.put(
		'/provider/edit/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		providerUsersAction.update
	)

	.get(
		'/provider/delete/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		providerUsersAction.destroy
	)

	.get(
		'/provider/list',
		adminUserMiddleware.checkToken,
		providerUsersAction.list
	)

	.get(
		'/provider/details/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		providerUsersAction.userDetails
	)

	.get(
		'/provider/export-excel',
		adminUserMiddleware.checkToken,
		providerUsersAction.exportExcel
	)

	.get(
		'/providers/count',
		adminUserMiddleware.checkToken,
		providerUsersAction.countTotalProviders
	)

	.post(
		'/driver/create',
		adminUserMiddleware.checkToken,
		driverUsersAction.create
	)

	.put(
		'/driver/edit/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		driverUsersAction.update
	)

	.get(
		'/driver/delete/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		driverUsersAction.destroy
	)

	.get(
		'/driver/list',
		adminUserMiddleware.checkToken,
		driverUsersAction.list
	)

	.get(
		'/drivers/count',
		adminUserMiddleware.checkToken,
		driverUsersAction.countTotalDrivers
	)

	.get(
		'/driver/details/:id',
		adminUserMiddleware.checkToken,
		param('id').isMongoId().withMessage('Params id must be valid mongo Id.'),
		handleValidationErrors,
		driverUsersAction.userDetails
	)

	.get(
		'/driver/export-excel',
		adminUserMiddleware.checkToken,
		driverUsersAction.exportExcel
	)

	.get(
		'/orders/list',
		adminUserMiddleware.checkToken,
		paymentAction.ordersList
	)

	.get(
		'/orders/sales-report',
		adminUserMiddleware.checkToken,
		paymentAction.ordersSalesReport
	)

	.get(
		'/orders/providers-export-excel/:id',
		adminUserMiddleware.checkToken,
		paymentAction.providersExportExcel
	)

	.get(
		'/orders/provider-sales-report/:id',
		adminUserMiddleware.checkToken,
		paymentAction.providerOrdersSalesReport
	)

	.get(
		'/order/details/:id',
		adminUserMiddleware.checkToken,
		paymentAction.orderDetails
	)

	.get(
		'/orders/count',
		adminUserMiddleware.checkToken,
		paymentAction.countOrdersFromAdmin
	)

module.exports = router